#!/usr/bin/env python
"""
CAN-ETH library
"""
# Copyright (c) 2016 proconX Pty Ltd. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. Neither the name of the author nor the names of its contributors may
#    be used to endorse or promote products derived from this software
#    without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
# FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
# THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
# BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
# OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
# AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
# THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.

__version__ = '1.0'

import struct

CANETH_PORT = 11898


class CanMsg():
    """Class which represents a CAN message"""

    def __init__(self, id = 0, bytes = (), ext = False, rtr = False, dlc = None):
        self.id = id
        if dlc:
            self.dlc = dlc
        else:
            self.dlc = len(bytes)
        self._bytes = list(bytes)
        self._bytes.extend([0] * (8 - len(self._bytes)))
        self.ext = ext
        self.rtr = rtr

    def __str__(self):
        if self.ext:
            s = '%07X' % (self.id)
        else:
            s = '%03X' % (self.id)
        if self.rtr:
            s = s + ' R'
        return s + ': ' + ' '.join(format(x, '02X') for x in self)

    def __repr__(self):
        return self.__str__()

    def __getitem__(self, index):
        if index < 0 or index > 8:
            raise IndexError("Index out of range")
        return self._bytes[index]

    def __setitem__(self, index, value):
        if index < 0 or index > 8:
            raise IndexError("Index out of range")
        self._bytes[index] = value

    def  __len__(self):
        return self.dlc

    def __iter__(self):
        self._index = 0
        return self

    def next(self):
        if self._index >= self.dlc:
            raise StopIteration
        byte = self._bytes[self._index]
        self._index = self._index + 1
        return byte

    def _validate(self):
        if self.id < 0 or (self.id > 0x7FF and not self.ext) or (self.id > 0x1FFFFFF and self.ext):
            raise IndexError("ID out of range")
        if self.dlc < 0 or self.dlc > 8:
            raise IndexError("DLC out of range")

    def pack(self):
        self._validate()
        return struct.pack('<IB8B??', self.id, self.dlc, self._bytes[0], self._bytes[1],
                           self._bytes[2], self._bytes[3], self._bytes[4], self._bytes[5],
                           self._bytes[6], self._bytes[7], self.ext, self.rtr)

    def unpack(self, data, offset = 0):
        (self.id, self.dlc, self._bytes[0], self._bytes[1], self._bytes[2],
         self._bytes[3], self._bytes[4], self._bytes[5], self._bytes[6],
         self._bytes[7], self.ext, self.rtr) = struct.unpack_from('<IB8B??', data, offset)
        self._validate()


class CanEthFrame():
    """Class which represents a CAN-ETH UDP frame"""

    MAX_DATAGRAM = 1482

    def __init__(self, data = None, version = 1):
        if data:
            self.unpack(data)
        else:
            self._messages = [] # v1 has up to 16 messages, v2 has up to 96
            self.version = version

    def __getitem__(self, index):
        if index < 0 or index > len(self._messages):
            raise IndexError("Index out of range")
        return self._messages[index]

    def __setitem__(self, index, value):
        if index < 0 or index > len(self._messages):
            raise IndexError("Index out of range")
        self._messages[index] = value

    def  __len__(self):
        return len(self._messages)

    def __iter__(self):
        self._index = 0
        return self

    def next(self):
        if self._index >= len(self._messages):
            raise StopIteration
        msg = self._messages[self._index]
        self._index = self._index + 1
        return msg

    def _validate(self, version, cnt):
        if (version == 1 and cnt > 16) or (version == 2 and cnt > 96):
            raise ValueError("Invalid message count")

    def append(self, msg):
        self._messages.append(msg)

    def pack(self):
        self._validate(self.version, len(self._messages))
        buffer = struct.pack('8sBB', 'ISO11898', self.version, len(self._messages))
        for msg in self._messages:
            buffer = buffer + msg.pack()
        return buffer + '\xFF' # EOL marker

    def unpack(self, data):
        magicId, version, cnt = struct.unpack_from('8sBB', data, 0)
        if magicId != 'ISO11898':
            raise ValueError("Invalid magic ID")
        self._validate(version, cnt)
        self._messages = []
        self.version = version
        for i in range(cnt):
            msg = CanMsg()
            msg.unpack(data, 10 + i * 15)
            self._messages.append(msg)


def echoTest():
    """A simple test program which echoes CAN messages with reversed bytes."""
    import socket
    sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    sock.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1);
    sock.bind(('', CANETH_PORT))
    while True:
        datagram, addr = sock.recvfrom(CanEthFrame.MAX_DATAGRAM)
        if datagram:
            print "UDP datagram received from CAN-ETH %s" % addr[0]
            for msg in CanEthFrame(datagram):
                print "CAN message received:", msg
                # Create a new CAN message to be replied
                replyMsg = CanMsg(dlc=msg.dlc, ext=msg.ext, rtr=msg.rtr)
                # Reverse ID (assumes a std ID)
                replyMsg.id = (msg.id & 0x00F) << 8 \
                              | (msg.id & 0x0F0) \
                              | (msg.id & 0xF00) >> 8
                # Reverse bytes
                for i in range(len(msg)):
                    replyMsg[i] = msg[len(msg) - 1 - i]
                # Create a new frame and add the modified message
                frame = CanEthFrame()
                frame.append(msg)
                # Send it back
                sock.sendto(frame.pack(), (addr[0], CANETH_PORT))
                print "CAN message sent:", replyMsg


if __name__ == "__main__":
    echoTest()

