/**
 * A simple UDP sender (broadcast) for the CAN-ETH gateway.
 *
 * Copyright (c) 2016 proconX Pty Ltd. All rights reserved.
 *
 * The following source file constitutes example program code and is
 * intended merely to illustrate useful programming techniques.  The user
 * is responsible for applying the code correctly.
 *
 * THIS SOFTWARE IS PROVIDED BY PROCONX AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL PROCONX OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef _WIN32
# include <winsock.h>
# pragma comment(lib, "Ws2_32.lib")
  WSADATA wsaData;
#else
# include <endian.h>
# include <sys/socket.h>
# include <netinet/in.h>
# include <arpa/inet.h>
# include <netdb.h>
#endif


#define CANETH_FRAME_VERSION           1
#define MAX_CAN_MSG                    16 // v1 has up to 16 messages, v2 has up to 96
#define UDP_PORT                       11898


/**
 * Structure of a CAN message
 *
 * @note Has to be packed to byte boundary!
 */
#pragma pack(1)
typedef struct
{
    unsigned int id;   // Little-endian format, max ID is 0x7FF if ext=0 or 0x1FFFFFF if ext=1
    unsigned char dlc; // Number of data bytes (0-8)
    unsigned char bytes[8];
    unsigned char ext; // 0 = 11-bit identifier, 1 = 29-bit identifier
    unsigned char rtr; // 0 = standard frame, 1 = remote tranmission request
} CanMsg;
#pragma pack()

/**
 * Structure of a CAN-ETH variable length UDP message.
 *
 * @note Has to be packed to byte boundary!
 */
#pragma pack(1)
typedef struct
{
    char magicId[8]; // Must be 'ISO11898'
    unsigned char version; // version 1 and 2 are supported
    unsigned char cnt; // v1 has up to 16 messages, v2 has up to 96
    CanMsg messages[MAX_CAN_MSG];
    char options[32]; // v1 has up to 128 option bytes, v2 has up to 32 only
} CanEthFrame;
#pragma pack()


/**
 * Init CAN-ETH UDP frame
 *
 * @param frame Pointer to CAN-ETH UDP frame
 *
 * @return int Length in bytes of CAN-ETH UDP frame
 */
int canEthFrameInit(CanEthFrame *frame)
{
   const char CANETH_MAGIC_ID[8] = {'I', 'S', 'O', '1', '1', '8', '9', '8'};

   memmove(&frame->magicId, CANETH_MAGIC_ID, sizeof(frame->magicId));
   frame->version = CANETH_FRAME_VERSION;
   frame->cnt = 0;
   frame->messages[0].id = 0xFFFFFFFF; // Set option mark to EOF

   return 10 + 1; // plus one option byte
}


/**
 * Pack a CAN message into a CAN-ETH UDP frame
 *
 * @param frame Pointer to CAN-ETH UDP frame
 * @param msg Pointer to a CAN message
 *
 * @return int Length in bytes of CAN-ETH UDP frame
 */
int canEthFramePack(CanEthFrame *frame, CanMsg *msg)
{
   if (frame->cnt < MAX_CAN_MSG)
   {
#if defined(__BYTE_ORDER) && defined(__BIG_ENDIAN) && __BYTE_ORDER == __BIG_ENDIAN
      // Swap ID bytes in case we run on a big-endian machine
      msg->id = le32toh(msg->id);
#endif
      memcpy(&frame->messages[frame->cnt], msg, sizeof(CanMsg));
      frame->cnt++;
      if (frame->cnt < MAX_CAN_MSG)
          frame->messages[frame->cnt].id = 0xFFFFFFFF; // Set option mark to EOF
      else
          frame->options[0] = (char) 0xFF; // EOF to mark no options contained in this frame
      return sizeof(CanEthFrame) - sizeof(CanMsg) * (MAX_CAN_MSG - frame->cnt);
   }
   else
      return -1;
}


CanEthFrame canEthFrame;


/**
 * Main entry point of application
 */
int main(int argc, char **argv)
{
    int result;
    struct sockaddr_in canEthAddr;
    int sock;
    int len;
    int option;
    CanMsg msg;


    //
    // Greeting
    //
    printf("CAN-ETH send example\n");

    //
    // Init UDP socket interface
    //
#ifdef _WIN32
    WSAStartup(0x0101, &wsaData); // Enable Windows' socket subsystem
#endif
    sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (sock == -1)
    {
        perror("Error opening UDP socket");
        return EXIT_FAILURE;
    }
    canEthAddr.sin_family = AF_INET;
    canEthAddr.sin_port = htons(UDP_PORT);
    canEthAddr.sin_addr.s_addr = htonl(INADDR_BROADCAST);
    option = 1; // Enable broadcasts on socket
    result = setsockopt(sock, SOL_SOCKET, SO_BROADCAST, (void *) &option, sizeof(option));
    if (result == -1)
    {
        perror("setsockopt failed");
        return EXIT_FAILURE;
    }

    // Init CAN-ETH UDP frame
    canEthFrameInit(&canEthFrame);

    //
    // Assemble CAN message
    //
    msg.id = 0x123;
    msg.ext = 0; // Standard identifier
    msg.rtr = 0; // Not a remote frame
    msg.dlc = 8;
    msg.bytes[0] = 0x11;
    msg.bytes[1] = 0x22;
    msg.bytes[2] = 0x33;
    msg.bytes[3] = 0x44;
    msg.bytes[4] = 0x55;
    msg.bytes[5] = 0x66;
    msg.bytes[6] = 0x77;
    msg.bytes[7] = 0x88;

    // Pack CAN message into UDP frame
    len = canEthFramePack(&canEthFrame, &msg);
    assert(len > 0);

    // Send UDP frame
    result = sendto(sock, (char *) &canEthFrame, len, 0, (struct sockaddr *) &canEthAddr, sizeof(canEthAddr));
    if (result == -1)
    {
       perror("UDP send failed");
       return EXIT_FAILURE;
    }

    printf("CAN message sent.\n");
    return EXIT_SUCCESS;
}
