/**
 * A simple UDP receiver for the CAN-ETH gateway.
 *
 * Copyright (c) 2016 proconX Pty Ltd. All rights reserved.
 *
 * The following source file constitutes example program code and is
 * intended merely to illustrate useful programming techniques.  The user
 * is responsible for applying the code correctly.
 *
 * THIS SOFTWARE IS PROVIDED BY PROCONX AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL PROCONX OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef _WIN32
# include <winsock.h>
# pragma comment(lib, "Ws2_32.lib")
  WSADATA wsaData;
#else
# include <sys/socket.h>
# include <netinet/in.h>
# include <arpa/inet.h>
# include <netdb.h>
#endif


#define CANETH_FRAME_VERSION           2
#define MAX_CAN_FRAMES                 96 // v1 has up to 16 messages, v2 has up to 96
#define UDP_PORT                       11898


/**
 * Structure of a CAN message
 *
 * @note Has to be packed to byte boundary!
 */
#pragma pack(1)
typedef struct
{
    unsigned int id;   // Little-endian format, max ID is 0x7FF if ext=0 or 0x1FFFFFF if ext=1
    unsigned char dlc; // Number of data bytes (0-8)
    unsigned char bytes[8];
    unsigned char ext; // 0 = 11-bit identifier, 1 = 29-bit identifier
    unsigned char rtr; // 0 = standard frame, 1 = remote tranmission request
} CanMsg;
#pragma pack()

/**
 * Structure of a CAN-ETH variable length UDP message.
 *
 * @note Has to be packed to byte boundary!
 */
#pragma pack(1)
typedef struct
{
    char magicId[8]; // Must be 'ISO11898'
    unsigned char version; // version 1 and 2 are supported
    unsigned char cnt; // v1 has up to 16 messages, v2 has up to 96
    CanMsg messages[MAX_CAN_FRAMES];
    char options[32]; // v1 has up to 128 option bytes, v2 has up to 32 only
} CanEthFrame;
#pragma pack()


/**
 * Validates a received CAN-ETH UDP frame
 *
 * @param frame Pointer to CAN-ETH UDP frame
 * @param len Number of bytes received
 *
 * @return int True if frame validates OK
 */
int canethFrameValidate(CanEthFrame *frame, int len)
{
    const char MAGIC_ISO11898_ID[8] = {'I', 'S', 'O', '1', '1', '8', '9', '8'};
    int i;

    // Check minimum length
    if (len <= 10)
        return 0;
    // Check for magic ID
    if (memcmp(&frame->magicId, MAGIC_ISO11898_ID, sizeof(MAGIC_ISO11898_ID)) != 0)
        return 0;
    // Check for supported version
    if (frame->version != 1 && frame->version != 2)
        return 0;
    // Check count within bounds
    if (frame->cnt > MAX_CAN_FRAMES)
        return 0;

#if defined(__BYTE_ORDER) && defined(__BIG_ENDIAN) && __BYTE_ORDER == __BIG_ENDIAN
    // Swap ID bytes in case we run on a big-endian machine
    for (i = 0; i < frame->cnt; i++)
        frame->messages[i].id = htole32(frame->messages[i].id);
#endif

    // All tests passed, OK
    return 1;
}


CanEthFrame canEthFrame;


/**
 * Main entry point of application
 */
int main(int argc, char **argv)
{
    int result;
    struct sockaddr_in canEthAddr;
    int sock;
    int len;


    //
    // Greeting
    //
    printf("CAN-ETH receive example\n");

    //
    // Init UDP socket interface
    //
#ifdef _WIN32
    WSAStartup(0x0101, &wsaData); // Enable Windows' socket subsystem
#endif
    sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (sock == -1)
    {
        perror("Error opening UDP socket");
        return EXIT_FAILURE;
    }
    canEthAddr.sin_family = AF_INET;
    canEthAddr.sin_port = htons(UDP_PORT);
    canEthAddr.sin_addr.s_addr = htonl(INADDR_ANY);
    result = bind(sock, (struct sockaddr *) &canEthAddr, sizeof (canEthAddr));
    if (result == -1)
    {
        perror("Error binding UDP socket");
        return EXIT_FAILURE;
    }

    //
    // Message process loop
    //
    printf("Listening to CAN messages...\n");
    for (;;)
    {
        int i, j;

        // Reveive a UDP packet
        len = recv(sock, &canEthFrame, sizeof(canEthFrame), 0);

        // Validate frame
        if (canethFrameValidate(&canEthFrame, len))
        {
           //
           // Extract CAN messages
           //
           for (i = 0; i < canEthFrame.cnt; i++)
           {
               CanMsg *msg = &canEthFrame.messages[i];

               printf("\r%0*X%s:", msg->ext ? 7 : 3, msg->id, msg->rtr ? " R":"");
               for (j = 0; j < msg->dlc; j++)
                   printf(" %02X", (unsigned int) msg->bytes[j]);
               printf("\n");
           }
        }
        else if (len < 0)
            perror("UDP reveive error");
    }

}
