/**
 * @file canlog.c
 *
 * A simple console based CAN message logger for the CAN-ETH gateway.
 *
 * @if NOTICE
 *
 * Copyright (c) 2012 proconX Pty Ltd. All rights reserved.
 *
 * The following source file constitutes example program code and is
 * intended merely to illustrate useful programming techniques.  The user
 * is responsible for applying the code correctly.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PROCONX AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL PROCONX OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @endif
 */

 
// Platform header
#include <stdio.h>
#include <stdlib.h>
#ifdef _WIN32
#  include <windows.h>
#else
#  include <unistd.h>
#endif

// Include CAN-ETH library header
#include "canethlib.h"


/**
 * CAN message receive notification handler
 *
 * @param hdl    Handle to identify CAN channel
 */
void receiveHandler(CanHandle hdl)
{
   CANFRAME msg;
   int rc;
   int i;

   // Retrieves a CAN message from the receive buffer
   rc = canEthRecv(hdl, &msg);
   if (rc == CANETH_SUCCESS)
   {
      printf("\r%lXh (%lud) %c%s:", msg.id, msg.id, msg.ext ? 'E':'S', msg.rtr ? "R":"");
      for (i = 0; i < msg.len; i++)
         printf(" %02Xh", (unsigned int) msg.byte[i]);
      printf("\n");
   }
}


/**
 * Main function
 *
 * @param argc Command line argument count
 * @param argv Command line argument value string array
 * @return Error code: 0 = OK, else error
 */
int main (int argc, char **argv)
{
   CanHandle hdl;
   int rc;

   // Check command line arguments
   if (argc != 2)
   {
       fprintf(stderr, "Usage: canlog ipaddr\n");
       return (EXIT_FAILURE);
   }

   // Init CAN-ETH library
   rc = canEthInit();
   if (rc == CANETH_SUCCESS)
   {
      // Open a CAN channel to a remote CAN-ETH gateway
      hdl = canEthOpen(argv[1]);
      if (hdl >= 0)
      {
         // Install a receive notification handler
         rc = canEthInstallRecvCallBack(hdl, receiveHandler);
         if (rc == CANETH_SUCCESS)
         {
            //
            // Listening to CAN messages
            //
            printf("Listening to CAN messages...\n");

            for (;;)
            {
               //
               // Print activity heartbeat
               //
#ifdef _WIN32
               Sleep(1000);
#else
               sleep(1);
#endif
               printf(".");
               fflush(stdout);
            }
         }
         else
            fprintf(stderr, "Error installing receive callback function (%d)!\n", rc);
      }
      else
         fprintf(stderr, "Error opening network connection to CAN-ETH (%d)!\n", hdl);
   }
   else
      fprintf(stderr, "Error starting receive background thread (%d)!\n", rc);

   // Shutdown the CAN-ETH Ethernet interface
   canEthShutdown();

	return EXIT_FAILURE;
}
