/**
 * @file canethlib.h
 *
 * CAN API for proconX CAN-ETH gateway - Header file
 *
 * @if NOTICE
 *
 * Copyright (c) 2011 proconX Pty Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PROCONX AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL PROCONX OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * For additional information see http://www.proconx.com
 *
 * @endif
 */


#ifndef _CANETHLIB_H_INCLUDED
#define _CANETHLIB_H_INCLUDED

#if defined(_WIN32)
#  define WIN32_LEAN_AND_MEAN
#  include <windows.h>
#else
#  include <inttypes.h>
#endif


/*****************************************************************************
 * Return codes
 *****************************************************************************/

/**
 * @name Error and Return Codes
 */
//@{

/**
 * Operation was successful
 *
 * This return codes indicates no error.
 */
#define CANETH_SUCCESS                 0

/**
 * Illegal argument error
 *
 * A parameter passed to the function returning this error code is invalid
 * or out of range.
 */
#define CANETH_ILLEGAL_ARGUMENT_ERROR  (-1)

/**
 * Illegal state error
 *
 * The function is called in a wrong state. This return code is returned by
 * all functions if the CAN channel has not been opened succesfully yet.
 */
#define CANETH_ILLEGAL_STATE_ERROR     (-2)

/**
 * No messages
 *
 * The receive buffer does not contain any messages.
 */
#define CANETH_NO_MSG                  (-3)

/**
 * Invalid IP address or hostname
 *
 * The IP address or hostname provided is invalid or cannot be
 * reached.
 */

#define CANETH_INVALID_HOST            (-4)

/**
 * Maximum number of open connections exceeded
 *
 * Maximum number of usable CAN connections were exceeded.
 */
#define CANETH_MAX_OPEN                (-5)

/**
 * Socket library error
 *
 * The TCP/IP socket library (e.g. WINSOCK) could not be loaded or the DLL
 * is missing or not installed.
 */
#define CANETH_SOCKET_LIB_ERROR        (-6)

/**
 * UDP port already bound
 *
 * Indicates that the specified UDP port cannot be bound. The
 * port might already be taken by another application or hasn't
 * been released yet by the TCP/IP stack for re-use.
 */
#define CANETH_PORT_ALREADY_BOUND      (-7)

/**
 * Creation of socket failed
 *
 * The creation of a UDP socket failed.
 */
#define CANETH_SOCKET_FAILED           (-8)

/**
 * Creation of background thread failed
 *
 * The system could not create the background receive thread.
 */
#define CANETH_CREATE_THREAD_FAILED    (-9)

//@}


/*****************************************************************************
 * Types
 *****************************************************************************/

/**
 * CAN message structure
 */
#if defined(_MSC_VER)
#  pragma pack(push)
#  pragma pack(1)
#elif defined(__GNUC__)
#  pragma pack(1)
#endif
typedef struct
{
#if defined (__DOXYGEN__)
   DWORD or uint32_t id; ///< CAN Identifier. Depending on platform
                         /// this is of DWORD (Win32) or uint32_t (gcc) type.
#elif defined(_WIN32)
   DWORD id;
#else
   uint32_t id;
#endif
   unsigned char len;        ///< Length of frame (Range 0 - 8)
   unsigned char byte[8];    ///< Message bytes
   unsigned char ext;        ///< Boolean flag to indicate extended frame
   unsigned char rtr;        ///< Boolean flag to indicate remote transmition bit
} CANFRAME;
#if defined(_MSC_VER)
#  pragma pack(pop)
#elif defined(__GNUC__)
#  pragma pack()
#endif

typedef int CanHandle;       ///< Handle to a CAN connection. Returned by canEthOpen().


/*****************************************************************************
 * Prototypes
 *****************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

int canEthInit(void);

void canEthShutdown(void);

CanHandle canEthOpen(const char * const hostName);

int canEthInstallRecvCallBack(CanHandle hdl, void (*f) (CanHandle hdl));

void canEthClose(CanHandle handle);

int canEthRecv(CanHandle hdl, CANFRAME *msg);

int canEthSendMsg(CanHandle hdl, const CANFRAME * const msg);

#ifdef __cplusplus
}
#endif


#endif // ifdef ..._H_INCLUDED

