/**
 * @file canethlib.cpp
 *
 * CAN API for proconX CAN-ETH gateway
 *
 * @if NOTICE
 *
 * Copyright (c) 2011 proconX Pty Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PROCONX AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL PROCONX OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * For additional information see http://www.proconx.com
 *
 * @endif
 */


#if defined(_WIN32)
#  define WIN32_LEAN_AND_MEAN
#  include <windows.h>
#  include <winsock.h>
#  pragma comment(lib, "Ws2_32.lib")
#  ifdef TRACE_TOOL
#     include "tracetool\tracetool.h"
#  endif
#else
#  include <inttypes.h>
#endif
#include "canethlib.h"


/*****************************************************************************
 * Definitions
 *****************************************************************************/

#define MAX_CAN_GATEWAYS               32 ///< Number of gateways supported

#define MAX_CAN_FRAMES                 16
#define ISO11898_FRAME_VERSION         1
#define OPTION_FRAMES                  1
#define UDP_PORT                       11898
char MAGIC_ISO11898_ID[] = {'I', 'S', 'O', '1', '1', '8', '9', '8'};


/*****************************************************************************
 * Types
 *****************************************************************************/

#if defined(_MSC_VER)
#  pragma pack(push)
#  pragma pack(1)
#elif defined(__GNUC__)
#  pragma pack(1)
#endif

typedef struct
{
   char magicId[8];
   unsigned char version;
   unsigned char cnt;
   CANFRAME canFrameArr[MAX_CAN_FRAMES];
   char optionArr[128];
} Iso11898Frame;

#if defined(_MSC_VER)
#  pragma pack(pop)
#elif defined(__GNUC__)
#  pragma pack()
#endif


#define CAN_BUF_SIZE 64
typedef struct
{
    CANFRAME data[CAN_BUF_SIZE];
    unsigned int size;
    unsigned int readIdx;
    unsigned int writeIdx;
    CRITICAL_SECTION criticalSection;
    sockaddr_in gatewayIpAddr;
    void (*recvNotifyFunc) (CanHandle hdl);
} CanBuffer;


/*****************************************************************************
 * Data
 *****************************************************************************/

static HANDLE recvThreadHdl;
static SOCKET recvSocket = INVALID_SOCKET;
static SOCKET sendSocket = INVALID_SOCKET;

static CanBuffer canBufArr[MAX_CAN_GATEWAYS];


/*****************************************************************************
 * Background threads
 *****************************************************************************/

/*
 * CAN background receive thread
 */
static DWORD WINAPI canRecvThread(LPVOID)
{
   for (;;)
   {
      int result;
      sockaddr_in senderAddr;
      int senderAddrSize;
      Iso11898Frame frame;

      senderAddrSize = sizeof(senderAddr);
      result = recvfrom(recvSocket, (char *) &frame, sizeof(frame), 0, (SOCKADDR *) &senderAddr, &senderAddrSize);

      // Correct frame ?
      if ((result > (sizeof(frame.magicId) + sizeof(frame.cnt) + sizeof(frame.version))) &&
          (frame.version == ISO11898_FRAME_VERSION) &&
          (memcmp(&frame.magicId, MAGIC_ISO11898_ID, sizeof(MAGIC_ISO11898_ID)) == 0))
      {
         int hdl;

         //
         // Search for correct communication endpoint
         //
         for (hdl = 0; hdl < MAX_CAN_GATEWAYS; hdl++)
         {
            int i = 0;
            void (*cb) (CanHandle hdl);

            EnterCriticalSection(&canBufArr[hdl].criticalSection);
            while ((i < (int) frame.cnt) &&
                  (canBufArr[hdl].size > 0) &&
                  (canBufArr[hdl].gatewayIpAddr.sin_family == senderAddr.sin_family) &&
                  (canBufArr[hdl].gatewayIpAddr.sin_addr.s_addr == senderAddr.sin_addr.s_addr))
            {
#if 0
#  ifdef TRACE_TOOL
               char buf[1000];
               sprintf(buf, "%d/%d: dlc=%d ext=%d rtr=%d %lX:%02X %02X %02X %02X %02X %02X %02X %02X",
                  i, (int) frame.cnt,
                  (int) frame.canFrameArr[i].len,
                       (int) frame.canFrameArr[i].ext, (int) frame.canFrameArr[i].rtr, frame.canFrameArr[i].id,
                       (unsigned int) frame.canFrameArr[i].byte[0], (unsigned int) frame.canFrameArr[i].byte[1],
                       (unsigned int) frame.canFrameArr[i].byte[2], (unsigned int) frame.canFrameArr[i].byte[3],
                       (unsigned int) frame.canFrameArr[i].byte[4], (unsigned int) frame.canFrameArr[i].byte[5],
                       (unsigned int) frame.canFrameArr[i].byte[6], (unsigned int) frame.canFrameArr[i].byte[7]);
               TTrace::Debug()->Send("canRecvThread", buf);
#  endif
#endif
               if (canBufArr[hdl].writeIdx == (canBufArr[hdl].readIdx + 1) % canBufArr[hdl].size)
               {
                  // Buffer overflow, also advance read index
                  canBufArr[hdl].readIdx = (canBufArr[hdl].readIdx + 1) % canBufArr[hdl].size;
               }
               memcpy(&canBufArr[hdl].data[canBufArr[hdl].writeIdx], &frame.canFrameArr[i], sizeof(canBufArr[hdl].data[canBufArr[hdl].writeIdx]));
               canBufArr[hdl].writeIdx = (canBufArr[hdl].writeIdx + 1) % canBufArr[hdl].size;
               cb = canBufArr[hdl].recvNotifyFunc;
               LeaveCriticalSection(&canBufArr[hdl].criticalSection);
               if (cb != NULL)
                  cb(hdl);
               i++;
               if (i >= (int) frame.cnt) // Leave inner loop? Then jump out of outer loop too...
                  goto done; // No need to do re-enter the critical section
               EnterCriticalSection(&canBufArr[hdl].criticalSection);
            }
            LeaveCriticalSection(&canBufArr[hdl].criticalSection);
         } // for hdl
      done: ;
      }
      else
      {
#ifdef TRACE_TOOL
         if (result < 0)
            TTrace::Debug()->Send("canRecvThread", "recvfrom failed");
#endif
         continue; // Discard frame
      }
   }
}


/*****************************************************************************
 * API functions
 *****************************************************************************/

/**
 * Initialises the CAN-ETH Ethernet interface.
 *
 * A background receive thread is created and started.
 *
 * @return CANETH_SUCCESS or a negative error code
 */
int canEthInit()
{
   int result;
   int hdl;
   WSADATA wsaData;
   DWORD recvThreadId;
   sockaddr_in recvAddr;

   //
   // Initialize Winsock
   //
   result = WSAStartup(0x0101, &wsaData);
   if (result != NO_ERROR)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthInit", "CANETH_SOCKET_LIB_ERROR");
#endif
      return CANETH_SOCKET_LIB_ERROR;
   }

   //
   // Create a receiver socket to receive datagrams
   //
   recvSocket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
   if (recvSocket == INVALID_SOCKET)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthInit", "CANETH_SOCKET_FAILED");
#endif
      return CANETH_SOCKET_FAILED;
   }

   //
   // Bind the socket to any address and the specified port
   //
   recvAddr.sin_family = AF_INET;
   recvAddr.sin_port = htons(UDP_PORT);
   recvAddr.sin_addr.s_addr = htonl(INADDR_ANY);

   result = bind(recvSocket, (SOCKADDR *) & recvAddr, sizeof (recvAddr));
   if (result != 0)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthInit", "CANETH_PORT_ALREADY_BOUND");
#endif
      closesocket(recvSocket);
      return CANETH_PORT_ALREADY_BOUND;
   }

   //
   // Create a socket for sending data
   //
   sendSocket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
   if (sendSocket == INVALID_SOCKET)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthInit", "CANETH_SOCKET_FAILED");
#endif
      closesocket(recvSocket);
      return CANETH_SOCKET_FAILED;
   }

   //
   // Init critical sections
   //
   for (hdl = 0; hdl < MAX_CAN_GATEWAYS; hdl++)
      InitializeCriticalSection(&canBufArr[hdl].criticalSection);

   //
   // Create worker thread
   //
   recvThreadHdl = CreateThread(NULL, 0, canRecvThread, 0, 0, &recvThreadId);
   if (recvThreadHdl == NULL)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthInit", "CANETH_CREATE_THREAD_FAILED");
#endif
      closesocket(recvSocket);
      closesocket(sendSocket);
      for (hdl = 0; hdl < MAX_CAN_GATEWAYS; hdl++)
         DeleteCriticalSection(&canBufArr[hdl].criticalSection);
      return CANETH_CREATE_THREAD_FAILED;
   }

   return CANETH_SUCCESS;
}


/**
 * Shutdown the CAN-ETH Ethernet interface.
 *
 * The background receive thread is terminated and all CAN channels closed.
 */
void canEthShutdown()
{
   int hdl;

   if (recvThreadHdl != NULL)
   {
      TerminateThread(recvThreadHdl, 0);
      recvThreadHdl = NULL;
      closesocket(recvSocket);
      closesocket(sendSocket);

      for (hdl = 0; hdl < MAX_CAN_GATEWAYS; hdl++)
      {
         canBufArr[hdl].size = 0;
         DeleteCriticalSection(&canBufArr[hdl].criticalSection);
      }
   }
   WSACleanup();
}


/**
 * Open a CAN channel to a remote CAN-ETH gateway
 *
 * @param hostName String with IP address or host name of the
 *                 remote CAN-ETH gateway.
 *
 * @return Handle to CAN channel or negative error code
 */
CanHandle canEthOpen(const char * const hostName)
{
   int hdl;
   sockaddr_in hostAddr;

   if (recvThreadHdl == NULL)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthOpen", "CANETH_ILLEGAL_STATE_ERROR");
#endif
      return CANETH_ILLEGAL_STATE_ERROR;
   }

   //
   // Determine host address
   //
   hostAddr.sin_family = AF_INET;
   hostAddr.sin_port = htons(UDP_PORT);
   hostAddr.sin_addr.s_addr = inet_addr((char *) hostName);
   if (hostAddr.sin_addr.s_addr == INADDR_NONE)
   {
      struct hostent *hostInfo;

      hostInfo = gethostbyname((char *) hostName);
      if (hostInfo == NULL)
      {
#ifdef TRACE_TOOL
         TTrace::Debug()->Send("canEthOpen", "CANETH_INVALID_HOST");
#endif
         return CANETH_INVALID_HOST;
      }
      hostAddr.sin_addr = *(struct in_addr *) hostInfo->h_addr;
   }

   //
   // Find free CAN communication channel
   //
   for (hdl = 0; hdl < MAX_CAN_GATEWAYS; hdl++)
   {
      if (canBufArr[hdl].size == 0)
      {
         //
         // Init receive buffer
         //
         EnterCriticalSection(&canBufArr[hdl].criticalSection);
         canBufArr[hdl].readIdx = 0;
         canBufArr[hdl].writeIdx = 0;
         canBufArr[hdl].gatewayIpAddr = hostAddr;
         canBufArr[hdl].recvNotifyFunc = NULL;
         canBufArr[hdl].size = CAN_BUF_SIZE;
         LeaveCriticalSection(&canBufArr[hdl].criticalSection);

         return hdl;
      }
   }
#ifdef TRACE_TOOL
   TTrace::Debug()->Send("canEthOpen", "CANETH_MAX_OPEN");
#endif
   return CANETH_MAX_OPEN;
}


/**
 * Install a receive notification handler.
 *
 * Must be called after canEthOpen
 *
 * @param hdl    Handle to identify CAN channel
 * @param f      Pointer to callback function
 *
 * @return CANETH_SUCCESS or a negative error code
 */
int canEthInstallRecvCallBack(CanHandle hdl, void (*f) (CanHandle hdl))
{
   if (hdl >= MAX_CAN_GATEWAYS)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthInstallRecvCallBack", "CANETH_ILLEGAL_ARGUMENT_ERROR");
#endif
      return CANETH_ILLEGAL_ARGUMENT_ERROR;
   }
   if (canBufArr[hdl].size == 0)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthInstallRecvCallBack", "CANETH_ILLEGAL_STATE_ERROR");
#endif
      return CANETH_ILLEGAL_STATE_ERROR;
   }

   EnterCriticalSection(&canBufArr[hdl].criticalSection);
   canBufArr[hdl].recvNotifyFunc = f;
   LeaveCriticalSection(&canBufArr[hdl].criticalSection);

   return CANETH_SUCCESS;
}


/**
 * Close a CAN receive channel
 *
 * @param hdl    Handle to identify CAN channel
 */
void canEthClose(CanHandle hdl)
{
   if (hdl < MAX_CAN_GATEWAYS)
   {
      EnterCriticalSection(&canBufArr[hdl].criticalSection);
      canBufArr[hdl].size = 0;
      LeaveCriticalSection(&canBufArr[hdl].criticalSection);
   }
}


/**
 * Send a CAN message
 *
 * @param hdl    Handle to identify CAN channel
 * @param msg    Pointer to CAN message to be sent
 *
 * @return CANETH_SUCCESS if sent
 */
int canEthSendMsg(CanHandle hdl, const CANFRAME * const msg)
{
   int result;
   char *optionPtr;
   int optionIdx;
   Iso11898Frame frame;

   if (hdl >= MAX_CAN_GATEWAYS)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthSendMsg", "CANETH_ILLEGAL_ARGUMENT_ERROR");
#endif
      return CANETH_ILLEGAL_ARGUMENT_ERROR;
   }
   if (canBufArr[hdl].size == 0)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthSendMsg", "CANETH_ILLEGAL_STATE_ERROR");
#endif
      return CANETH_ILLEGAL_STATE_ERROR;
   }

   //
   // Pack CAN frames into Ethernet frame
   //
   frame.version = ISO11898_FRAME_VERSION;
   frame.cnt = 1;
   memmove(&frame.magicId, MAGIC_ISO11898_ID, sizeof(frame.magicId));
   memmove(&frame.canFrameArr[0], msg, sizeof(frame.canFrameArr[0]));

   //
   // Add options
   //
   optionPtr = (char *) &frame.canFrameArr[frame.cnt];
   optionIdx = 0;
   // Size option
   optionPtr[optionIdx++] = OPTION_FRAMES;
   optionPtr[optionIdx++] = MAX_CAN_FRAMES;
   // End of options mark
   optionPtr[optionIdx++] = (char) 0xFF; // EOF

   //
   // Send Ethernet frame
   //
   result = sendto(sendSocket, (char *) &frame,
                   optionIdx + sizeof(frame.magicId) + sizeof(frame.cnt) + sizeof(frame.version) +
                   frame.cnt * sizeof(frame.canFrameArr[0]), 0,
                   (SOCKADDR *) &canBufArr[hdl].gatewayIpAddr,
                   sizeof(canBufArr[hdl].gatewayIpAddr));
#if 0
#  ifdef TRACE_TOOL
      char buf[1000];
      sprintf(buf, "dlc=%d flags=%d/%d %lX:%02X %02X %02X %02X %02X %02X %02X %02X",
         msg->len, msg->ext, msg->rtr, msg->id,
         (unsigned int) msg->byte[0], (unsigned int) msg->byte[1],
         (unsigned int) msg->byte[2], (unsigned int) msg->byte[3],
         (unsigned int) msg->byte[4], (unsigned int) msg->byte[5],
         (unsigned int) msg->byte[6], (unsigned int) msg->byte[7]);
      TTrace::Debug()->Send("canEthSendMsg", buf);
#  endif
#endif

   return CANETH_SUCCESS;
}


/**
 * Retrieves a CAN message from the receive buffer. If no
 * messages if available, CANETH_NO_MSG is returned.
 *
 * @param hdl    Handle to identify CAN channel
 * @param msg    Pointer to container for received CAN message
 *
 * @return CANETH_SUCCESS if OK or CANETH_NO_MSG if no message
 *         received.
 */
int canEthRecv(CanHandle hdl, CANFRAME *msg)
{
   if (hdl >= MAX_CAN_GATEWAYS)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthRecv", "CANETH_ILLEGAL_ARGUMENT_ERROR");
#endif
      return CANETH_ILLEGAL_ARGUMENT_ERROR;
   }
   if (canBufArr[hdl].size == 0)
   {
#ifdef TRACE_TOOL
      TTrace::Debug()->Send("canEthRecv", "CANETH_ILLEGAL_STATE_ERROR");
#endif
      return CANETH_ILLEGAL_STATE_ERROR;
   }

   EnterCriticalSection(&canBufArr[hdl].criticalSection);
   if (canBufArr[hdl].readIdx == canBufArr[hdl].writeIdx)
   {
      LeaveCriticalSection(&canBufArr[hdl].criticalSection);
      return CANETH_NO_MSG;
   }
   *msg = canBufArr[hdl].data[canBufArr[hdl].readIdx];
   canBufArr[hdl].readIdx = (canBufArr[hdl].readIdx + 1) % canBufArr[hdl].size;
   LeaveCriticalSection(&canBufArr[hdl].criticalSection);
#if 0
#  ifdef TRACE_TOOL
      char buf[1000];
      sprintf(buf, "dlc=%d flags=%d/%d %lX:%02X %02X %02X %02X %02X %02X %02X %02X",
         msg->len, msg->ext, msg->rtr, msg->id,
         (unsigned int) msg->byte[0], (unsigned int) msg->byte[1],
         (unsigned int) msg->byte[2], (unsigned int) msg->byte[3],
         (unsigned int) msg->byte[4], (unsigned int) msg->byte[5],
         (unsigned int) msg->byte[6], (unsigned int) msg->byte[7]);
      TTrace::Debug()->Send("canEthRecv", buf);
#  endif
#endif
   return CANETH_SUCCESS;
}
